package com.mangoslab.nemonicsdktest;

import static com.mangoslab.nemonicsdktest.SelectPrinterActivity.PREF_SELECTED_PRINTER_MAC_ADDRESS;
import static com.mangoslab.nemonicsdktest.SelectPrinterActivity.PREF_SELECTED_PRINTER_NAME;
import static com.mangoslab.nemonicsdktest.SelectPrinterActivity.PREF_SELECTED_PRINTER_TYPE;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.SwitchCompat;

import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.preference.PreferenceManager;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.RadioButton;
import android.widget.TextView;
import android.widget.Toast;

import com.mangoslab.nemonicsdk.INPrinterControllerCallback;
import com.mangoslab.nemonicsdk.NPrintInfo;
import com.mangoslab.nemonicsdk.NPrinter;
import com.mangoslab.nemonicsdk.NPrinterController;
import com.mangoslab.nemonicsdk.constants.NBatteryStatus;
import com.mangoslab.nemonicsdk.constants.NCartridgeType;
import com.mangoslab.nemonicsdk.constants.NConnectState;
import com.mangoslab.nemonicsdk.constants.NPrintQuality;
import com.mangoslab.nemonicsdk.constants.NPrinterStatus;
import com.mangoslab.nemonicsdk.constants.NPrinterType;
import com.mangoslab.nemonicsdk.constants.NResult;
import com.mangoslab.nemonicsdk.constants.NResultString;

import java.io.FileNotFoundException;
import java.io.InputStream;

public class MainActivity
        extends AppCompatActivity
        implements View.OnClickListener,
        INPrinterControllerCallback {
    private static final String TAG = "MainActivity";

    private ImageView iv_image;

    private TextView tv_selected_printer_name;
    private Button bt_select_printer;
    private Button bt_connect_disconnect;
    private Button bt_select_image;

    private EditText et_copies;
    private Button bt_print;

    private RadioButton rb_low_fast;
    private RadioButton rb_middle;
    private RadioButton rb_high_slow;

    private SwitchCompat sw_last_page_cut;
    private SwitchCompat sw_dither;

    private Button bt_template_set;
    private Button bt_template_clear;

    private TextView tv_printer_status;
    private Button bt_printer_status_get;
    private TextView tv_cartridge_type;
    private Button bt_cartridge_type_get;
    private TextView tv_printer_name;
    private Button bt_printer_name_get;

    private TextView tv_battery_level;
    private Button bt_battery_level_get;
    private TextView tv_battery_status;
    private Button bt_battery_status_get;


    private ActivityResultLauncher<Intent> selectImageLauncher;
    private ActivityResultLauncher<Intent> selectPrinterLauncher;

    private NPrinter printer = null;
    private NPrinterController printerController = null;

    private Bitmap selectedImage = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        printerController = new NPrinterController(this, this);

        InitView();
        initActivityResult();
    }

    @Override
    protected void onResume() {
        super.onResume();
        updateSelectedPrinter();
    }

    private int backPressedNum = 0;
    @Override
    public void onBackPressed() {
        if (backPressedNum == 0) {
            backPressedNum++;
            Handler handler = new Handler();
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    backPressedNum = 0;
                }
            }, 2000);
            showToast("If you want to exit, press back 1 more time.");
        }
        else {
            if (printerController.getConnectState() != NConnectState.DISCONNECTED.getValue()) {
                printerController.disconnect();
                bt_connect_disconnect.setText("Disconnected");
            }
            super.onBackPressed();
        }
    }

    private void InitView() {
        iv_image = findViewById(R.id.iv_image);

        tv_selected_printer_name = findViewById(R.id.tv_selected_printer_name);
        bt_select_printer = findViewById(R.id.bt_select_printer);
        bt_connect_disconnect = findViewById(R.id.bt_connect_disconnect);
        bt_select_image = findViewById(R.id.bt_select_image);

        et_copies = findViewById(R.id.et_copies);
        bt_print = findViewById(R.id.bt_print);

        rb_low_fast = findViewById(R.id.rb_low_fast);
        rb_middle = findViewById(R.id.rb_middle);
        rb_high_slow = findViewById(R.id.rb_high_slow);

        sw_last_page_cut = findViewById(R.id.sw_last_page_cut);
        sw_dither = findViewById(R.id.sw_dither);

        bt_template_set = findViewById(R.id.bt_template_set);
        bt_template_clear = findViewById(R.id.bt_template_clear);

        tv_printer_status = findViewById(R.id.tv_printer_status);
        bt_printer_status_get = findViewById(R.id.bt_printer_status_get);
        tv_cartridge_type = findViewById(R.id.tv_cartridge_type);
        bt_cartridge_type_get = findViewById(R.id.bt_cartridge_type_get);
        tv_printer_name = findViewById(R.id.tv_printer_name);
        bt_printer_name_get = findViewById(R.id.bt_printer_name_get);

        tv_battery_level = findViewById(R.id.tv_battery_level);
        bt_battery_level_get = findViewById(R.id.bt_battery_level_get);
        tv_battery_status = findViewById(R.id.tv_battery_status);
        bt_battery_status_get = findViewById(R.id.bt_battery_status_get);


        bt_select_printer.setOnClickListener(this);
        bt_connect_disconnect.setOnClickListener(this);
        bt_select_image.setOnClickListener(this);
        bt_print.setOnClickListener(this);

        bt_template_set.setOnClickListener(this);
        bt_template_clear.setOnClickListener(this);

        bt_printer_status_get.setOnClickListener(this);
        bt_cartridge_type_get.setOnClickListener(this);
        bt_printer_name_get.setOnClickListener(this);

        bt_battery_level_get.setOnClickListener(this);
        bt_battery_status_get.setOnClickListener(this);
    }

    private void initActivityResult() {
        selectImageLauncher = registerForActivityResult(
                new ActivityResultContracts.StartActivityForResult(),
                result -> {
                    int resultCode = result.getResultCode();
                    Intent intent = result.getData();
                    if (resultCode == RESULT_OK && intent != null) {
                        selectedImage(intent.getData());
                    }
                    else {
                        setResult("Cannot bind image.");
                    }
                }
        );

        selectPrinterLauncher = registerForActivityResult(
                new ActivityResultContracts.StartActivityForResult(),
                result -> {
                    updateSelectedPrinter();
                }
        );
    }

    @Override
    public void onClick(View v) {
        switch (v.getId()) {
            case R.id.bt_select_printer:
                clickSelectPrinter();
                break;
            case R.id.bt_connect_disconnect:
                clickConnectDisconnect();
                break;
            case R.id.bt_select_image:
                clickSelectImage();
                break;
            case R.id.bt_print:
                clickPrint();
                break;

            case R.id.bt_printer_status_get:
                clickPrinterStatusGet();
                break;
            case R.id.bt_cartridge_type_get:
                clickCartridgeTypeGet();
                break;
            case R.id.bt_printer_name_get:
                clickPrinterNameGet();
                break;
            case R.id.bt_template_set:
                clickTemplateSet();
                break;
            case R.id.bt_template_clear:
                clickTemplateClear();
                break;
            case R.id.bt_battery_level_get:
                clickBatteryLevelGet();
                break;
            case R.id.bt_battery_status_get:
                clickBatteryStatusGet();
                break;
        }
    }

    private void clickSelectPrinter() {
        Intent intent = new Intent(getApplicationContext(), SelectPrinterActivity.class);
        selectPrinterLauncher.launch(intent);
    }

    private void clickConnectDisconnect() {
        String btName = bt_connect_disconnect.getText().toString();

        if (printer == null) {
            showToast("No selected printer!");
            return ;
        }

        if (btName.equals("Connect")) {
            int result = printerController.connect(printer);

            if (result == NResult.OK)
                bt_connect_disconnect.setText("Disconnect");
            else
                setResult(String.format("Connect failed(%d)", result));
        }
        else {
            printerController.disconnect();
            bt_connect_disconnect.setText("Connect");
        }
    }

    private void clickSelectImage() {
        Intent intent = new Intent();
        intent.setType("image/*");
        intent.setAction(Intent.ACTION_GET_CONTENT);
        selectImageLauncher.launch(intent);
    }

    private void clickPrint() {
        int copies = 1;

        try {
            copies = Integer.parseInt(et_copies.getText().toString());
        } catch (NumberFormatException ignored) {
            showToast("Please, enter a number(Decimal)");
            return;
        }

        if (copies <= 0) {
            showToast("Please, enter a positive number");
            return;
        }

        if (selectedImage == null) {
            showToast("No selected image!");
            return;
        }

        NPrintInfo printInfo = new NPrintInfo(printer, selectedImage);
        printInfo.setCopies(copies)
                .setEnableLastPageCut(sw_last_page_cut.isChecked())
                .setEnableDither(sw_dither.isChecked());
        if (rb_low_fast.isChecked())
            printInfo.setPrintQuality(NPrintQuality.LOW_FAST);
        else if (rb_middle.isChecked())
            printInfo.setPrintQuality(NPrintQuality.MIDDLE);
        else
            printInfo.setPrintQuality(NPrintQuality.HIGH_SLOW);
        int printResult = printerController.print(printInfo);

        switch (printResult) {
            case NResult.OK:
                setResult("Success");
                break;
            case NResult.NOT_CONNECTED:
                showToast("Not connected");
                break;
            default:
                setResult(String.format("%d", printResult));
                break;
        }
    }

    private void clickPrinterStatusGet() {
        tv_printer_status.setText("");
        int result = printerController.getPrinterStatus();

        if (result >= 0) {
            NPrinterStatus printerStatus = NPrinterStatus.valueOf(result);
            switch (printerStatus) {
                case OK:
                    tv_printer_status.setText("OK");
                    setResult("OK");
                    break;
                case OUT_OF_PAPER:
                    tv_printer_status.setText("Out of paper");
                    setResult("Out of paper");
                    break;
                case COVER_OPENED:
                    tv_printer_status.setText("Cover opened");
                    setResult("Cover opened");
                    break;
                case OVERHEAT:
                    tv_printer_status.setText("Overheat");
                    setResult("Overheat");
                    break;
                case PAPER_JAM:
                    tv_printer_status.setText("Paper jam");
                    setResult("Paper jam");
                    break;
            }
        }
        else
            processResult(result);
    }

    private void clickCartridgeTypeGet() {
        tv_cartridge_type.setText("");
        int result = printerController.getCartridgeType();

        if (result >= 0) {
            NCartridgeType cartridgeType = NCartridgeType.valueOf(result);
            switch (cartridgeType) {
                case WHITE:
                    tv_cartridge_type.setText("White(0)");
                    setResult("White");
                    break;
                case YELLOW:
                    tv_cartridge_type.setText("Yellow(2)");
                    setResult("Yellow");
                    break;
                case GREEN:
                    tv_cartridge_type.setText("Green(3)");
                    setResult("Green");
                    break;
                case BLUE:
                    tv_cartridge_type.setText("Blue(4)");
                    setResult("Blue");
                    break;
                case PINK:
                    tv_cartridge_type.setText("Pink(5)");
                    setResult("Pink");
                    break;
                case L1:
                    tv_cartridge_type.setText("L1(6)");
                    setResult("L1");
                    break;
                case L2:
                    tv_cartridge_type.setText("L2(7)");
                    setResult("L2");
                    break;
                case L3:
                    tv_cartridge_type.setText("L3(8)");
                    setResult("L3");
                    break;
                case L4:
                    tv_cartridge_type.setText("L4(9)");
                    setResult("L4");
                    break;
                case M1:
                    tv_cartridge_type.setText("M1(6)");
                    setResult("M1");
                    break;
                case M2:
                    tv_cartridge_type.setText("M2(7)");
                    setResult("M2");
                    break;
                case M3:
                    tv_cartridge_type.setText("M3(8)");
                    setResult("M3");
                    break;
                case M4:
                    tv_cartridge_type.setText("M4(9)");
                    setResult("M4");
                    break;
                case NONE:
                    setResult("Failed");
                    break;
            }
        }
        else
            processResult(result);
    }

    private void clickPrinterNameGet() {
        tv_printer_name.setText("");
        NResultString result = printerController.getPrinterName();

        if (result.getResult() == NResult.OK) {
            String printerName = result.getValue();
            tv_printer_name.setText(printerName);
            setResult("Success");
        }
        else
            processResult(result.getResult());
    }

    private void clickTemplateSet() {
        int result = printerController.setTemplate(selectedImage, true, sw_dither.isChecked());

        processResult(result);
    }

    private void clickTemplateClear() {
        int result = printerController.clearTemplate();

        processResult(result);
    }

    private void clickBatteryLevelGet() {
        tv_battery_level.setText("");
        int result = printerController.getBatteryLevel();

        if (result >= 0) {
            tv_battery_level.setText(String.format("%d", result));
            setResult("Success");
        }
        else
            processResult(result);
    }

    private void clickBatteryStatusGet() {
        tv_battery_status.setText("");
        int result = printerController.getBatteryStatus();

        if (result >= 0) {
            NBatteryStatus batteryStatus = NBatteryStatus.valueOf(result);
            switch (batteryStatus) {
                case NO_CHARGING:
                    tv_battery_status.setText("Normal (No charging)");
                    setResult("Normal (No charging)");
                    break;
                case LOW_NO_CHARGING:
                    tv_battery_status.setText("Low (No charging)");
                    setResult("Low (No charging)");
                    break;
                case CHARGING:
                    tv_battery_status.setText("Normal (Charging)");
                    setResult("Normal (Charging)");
                    break;
                case LOW_CHARGING:
                    tv_battery_status.setText("Low (Charging)");
                    setResult("Low (Charging)");
                    break;
                default:
                    break;
            }
        }
        else
            processResult(result);
    }

    private void processResult(int result) {
        switch (result) {
            case NResult.OK:
                setResult("Success");
                break;
            case NResult.UNKNOWN:
                setResult("Unknown");
                break;
            case NResult.TIMEOUT:
                setResult("Timeout");
                break;
            case NResult.CANCELED:
                setResult("canceled");
                break;
            case NResult.NOT_CONNECTED:
                setResult("Not connected");
                break;
            case NResult.ALREADY_CONNECTED:
                setResult("Already connected");
                break;
            case NResult.SOCKET_ERROR:
                setResult("Socket error");
                break;
            case NResult.CONNECT_ERROR:
                setResult("connect error");
                break;
            case NResult.CONNECT_FAILED:
                setResult("Connect failed");
                break;
            case NResult.SESSION_ERROR:
                setResult("Session error");
                break;
            case NResult.IO_RECEIVE_ERROR:
                setResult("IO receive error");
                break;
            case NResult.IO_SEND_ERROR:
                setResult("IO send error");
                break;
            case NResult.CONNECT_SERVICE_NOT_FOUND:
                setResult("Connect service not found");
                break;
            case NResult.BLUETOOTH_UNSUPPORTED:
                setResult("Bluetooth unsupported");
                break;
            case NResult.BLUETOOTH_DISABLED:
                setResult("Bluetooth disabled");
                break;
            case NResult.BLUETOOTH_NO_PERMISSION:
                setResult("Bluetooth no permission");
                break;
            case NResult.LOCATION_DISABLED:
                setResult("location disabled");
                break;
            case NResult.LOCATION_NO_PERMISSION:
                setResult("location no permission");
                break;
            case NResult.INVALID_PARAMETER:
                setResult("Not valid parameter");
                break;
            case NResult.NOT_MATCHED_PRINTER_TYPE:
                setResult("Not matched printer type");
                break;
            case NResult.NO_CALLBACK:
                setResult("No callback");
                break;
            case NResult.SEND_FAILED:
                setResult("Send failed");
                break;
            case NResult.BATTERY_LOW:
                setResult("Battery low");
                break;
            case NResult.NOT_MATCHED_COMMAND_RESULT_FORMAT:
                setResult("Not matched command result format");
                break;
            case NResult.INVALID_PRINTER_NAME:
                setResult("Not valid printer name");
                break;
            case NResult.INVALID_PRINTER_RESULT:
                setResult("Not valid printer result");
                break;
            case NResult.PRINTER_RESULT_FAILED:
                setResult("Printer result failed");
                break;
            default:
                setResult(String.format("Unknown(%d)", result));
                break;
        }
    }

    @Override
    public void disconnected() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                bt_connect_disconnect.setText("Connect");
                showToast("Disconnected");
            }
        });
    }

    @Override
    public void printProgress(int index, int total, int result) {
    }

    @Override
    public void printComplete(int result) {
    }

    private void updateSelectedPrinter() {
        SharedPreferences pref = PreferenceManager.getDefaultSharedPreferences(this);
        String name = pref.getString(PREF_SELECTED_PRINTER_NAME, null);
        String macAddress = pref.getString(PREF_SELECTED_PRINTER_MAC_ADDRESS, null);
        int type = pref.getInt(PREF_SELECTED_PRINTER_TYPE, NPrinterType.NONE.getValue());

        if (name != null
                && macAddress != null)
            printer = new NPrinter(NPrinterType.valueOf(type), name, macAddress);
        else
            printer = null;

        if (printer != null) {
            name = printer.getName();
            macAddress = printer.getMacAddress();
            tv_selected_printer_name.setText(String.format("%s (%s)", name, macAddress));
        }
        else
            tv_selected_printer_name.setText("No selected printer");
    }

    private void selectedImage(Uri uri) {
        try {
            InputStream inputStream = getContentResolver().openInputStream(uri);
            selectedImage = BitmapFactory.decodeStream(inputStream);
            int width = selectedImage.getWidth();
            int height = selectedImage.getHeight();
            if (width != 576)
                selectedImage = Bitmap.createScaledBitmap(selectedImage, 576, height * 576 / width, true);
            iv_image.setImageBitmap(selectedImage);
        }
        catch (FileNotFoundException e) {
            e.printStackTrace();
        }
    }

    private void setResult(final String message) {
        showToast(message);
    }

    private void showToast(final String message) {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                Toast.makeText(MainActivity.this, message, Toast.LENGTH_SHORT).show();
            }
        });
    }
}