package com.mangoslab.nemonicsdktest;

import android.content.SharedPreferences;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.view.View;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;

import com.mangoslab.nemonicsdk.INPrinterScanControllerCallback;
import com.mangoslab.nemonicsdk.NPrinter;
import com.mangoslab.nemonicsdk.NPrinterScanController;
import com.mangoslab.nemonicsdk.constants.NPrinterType;
import com.mangoslab.nemonicsdk.constants.NResult;

public class SelectPrinterActivity
        extends AppCompatActivity
        implements View.OnClickListener,
        INPrinterScanControllerCallback,
        AdapterView.OnItemClickListener {
    private static final String TAG = "SelectPrinterActivity";

    public static final String PREF_SELECTED_PRINTER_NAME = "prefSelectedPrinterName";
    public static final String PREF_SELECTED_PRINTER_MAC_ADDRESS = "prefSelectedPrinterMacAddress";
    public static final String PREF_SELECTED_PRINTER_TYPE = "prefSelectedPrinterType";

    private Button bt_back;
    private TextView tv_selected_printer_name;
    private Button bt_unselect;
    private ListView lv_printers;
    private Button bt_refresh;

    private PrinterListAdapter printerListAdapter;

    private NPrinter selectedPrinter = null;

    private NPrinterScanController printerScanController = null;

    private boolean isInScan = false;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_select_printer);

        initView();

        printerScanController = new NPrinterScanController(this, this);
        int scanResult = printerScanController.startScan();
        switch (scanResult) {
            case NResult.BLUETOOTH_UNSUPPORTED:
                break;
            case NResult.BLUETOOTH_DISABLED:
            case NResult.LOCATION_DISABLED:
                break;
            case NResult.BLUETOOTH_NO_PERMISSION:
            case NResult.LOCATION_NO_PERMISSION:
                showToast("No required permission");
                break;
        }
    }

    @Override
    public void onBackPressed() {
        printerScanController.stopScan();
        super.onBackPressed();
    }

    @Override
    protected void onResume() {
        super.onResume();

        SharedPreferences pref = PreferenceManager.getDefaultSharedPreferences(this);
        String name = pref.getString(PREF_SELECTED_PRINTER_NAME, null);
        String macAddress = pref.getString(PREF_SELECTED_PRINTER_MAC_ADDRESS, null);
        int type = pref.getInt(PREF_SELECTED_PRINTER_TYPE, NPrinterType.NONE.getValue());

        if (name != null
                && macAddress != null)
            selectedPrinter = new NPrinter(NPrinterType.valueOf(type), name, macAddress);
        else
            selectedPrinter = null;

        updateSelectedPrinter();
    }

    @Override
    protected void onPause() {
        super.onPause();

        SharedPreferences pref = PreferenceManager.getDefaultSharedPreferences(this);
        SharedPreferences.Editor editor = pref.edit();
        if (selectedPrinter != null) {
            editor.putString(PREF_SELECTED_PRINTER_NAME, selectedPrinter.getName());
            editor.putString(PREF_SELECTED_PRINTER_MAC_ADDRESS, selectedPrinter.getMacAddress());
            editor.putInt(PREF_SELECTED_PRINTER_TYPE, selectedPrinter.getType().getValue());
        } else {
            editor.remove(PREF_SELECTED_PRINTER_NAME);
            editor.remove(PREF_SELECTED_PRINTER_MAC_ADDRESS);
            editor.remove(PREF_SELECTED_PRINTER_TYPE);
        }
        editor.apply();
    }

    @Override
    public void deviceFound(NPrinter printer) {
        printerListAdapter.add(new PrinterListItem(printer));
    }

    private void initView() {
        bt_back = findViewById(R.id.bt_back);
        tv_selected_printer_name = findViewById(R.id.tv_selected_printer_name);
        bt_unselect = findViewById(R.id.bt_unselect);
        lv_printers = findViewById(R.id.lv_printers);
        bt_refresh = findViewById(R.id.bt_refresh);

        printerListAdapter = new PrinterListAdapter(this);
        lv_printers.setAdapter(printerListAdapter);
        lv_printers.setOnItemClickListener(this);

        bt_back.setOnClickListener(this);
        bt_unselect.setOnClickListener(this);
        bt_refresh.setOnClickListener(this);
    }

    @Override
    public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
        PrinterListItem item = printerListAdapter.get(position);
        selectedPrinter = item.getPrinter();
        updateSelectedPrinter();
    }

    @Override
    public void onClick(View v) {
        switch (v.getId()) {
            case R.id.bt_back:
                clickBack();
                break;
            case R.id.bt_unselect:
                clickUnselect();
                break;
            case R.id.bt_refresh:
                clickRefresh();
                break;
        }
    }

    private void clickBack() {
        onBackPressed();
    }

    private void clickUnselect() {
        selectedPrinter = null;
        tv_selected_printer_name.setText("No selected printer");
    }

    private void clickRefresh() {
        printerListAdapter.reset();
        int result = printerScanController.startScan();

        switch (result) {
            case NResult.BLUETOOTH_UNSUPPORTED:
                break;
            case NResult.BLUETOOTH_DISABLED:
            case NResult.LOCATION_DISABLED:
                break;
            case NResult.BLUETOOTH_NO_PERMISSION:
            case NResult.LOCATION_NO_PERMISSION:
                showToast("No required permission");
                break;
        }
    }

    private void updateSelectedPrinter() {
        if (selectedPrinter != null) {
            String name = selectedPrinter.getName();
            String macAddress = selectedPrinter.getMacAddress();
            tv_selected_printer_name.setText(String.format("%s (%s)", name, macAddress));
        }
        else
            tv_selected_printer_name.setText("No selected printer");
    }

    private void showToast(String message) {
        Toast.makeText(this, message, Toast.LENGTH_SHORT).show();
    }
}
