//
//  SelectPrinterViewController.swift
//  NemonicSdkTest
//
//  Created by Yeongjae Kim on 2022/10/21.
//

import Foundation
import UIKit
import NemonicSdk
import Toast_Swift

class SelectPrinterViewController: UIViewController {
    
    static let KEY_NAME = "selectedPrinterName"
    static let KEY_MAC_ADDRESS = "selectedPrinterMacAddress"
    static let KEY_TYPE = "selectedPrinterType"
        
    @IBOutlet weak var selectedPrinterNameLabel: UILabel!
    @IBOutlet weak var unselect: UIButton!
    
    @IBOutlet weak var searchedPrinters: UITableView!
    
    private var selectedPrinter: NPrinter?
    private var printerScanController: NPrinterScanController?
    
    private var printers = [NPrinter]()
    
    override func viewDidLoad() {
        super.viewDidLoad()
        printerScanController = NPrinterScanController(self)
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        let printer = SelectPrinterViewController.getSelectedPrinter()
        updateSelectedPrinter(printer)
        
        Thread.init() {
            self.startScan()
        }.start()
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        resetPrinter()
        stopScan()
    }
    
    @IBAction func actionBack(_ sender: Any) {
        self.dismiss(animated: true)
    }
    
    @IBAction func actionUnselect(_ sender: Any) {
        updateSelectedPrinter(nil)
    }
    
    @IBAction func actionRefresh(_ sender: Any) {
        stopScan()
        resetPrinter()
        startScan()
    }
    
    private func startScan() {
        if printerScanController != nil {
            let result = printerScanController!.startScan()
        }
    }
    
    private func stopScan() {
        printerScanController?.stopScan()
    }
    
    private func addPrinter(_ printer: NPrinter) {
        printers.append(printer)
        DispatchQueue.main.async {
            self.searchedPrinters.reloadData();
        }
    }
    
    private func updateSelectedPrinter(_ printer: NPrinter?) {
        let defaults = UserDefaults.standard
        
        if printer != nil {
            let name = printer!.getName()
            let macAddress = printer!.getMacAddress()
            let type = printer!.getType().rawValue
            selectedPrinterNameLabel.text = name + "(" + macAddress + ")"
            unselect.isHidden = false
            defaults.setValue(name, forKey: SelectPrinterViewController.KEY_NAME)
            defaults.setValue(macAddress, forKey: SelectPrinterViewController.KEY_MAC_ADDRESS)
            defaults.setValue(type, forKey: SelectPrinterViewController.KEY_TYPE)
        }
        else {
            selectedPrinterNameLabel.text = "No selected printer"
            unselect.isHidden = true
            defaults.removeObject(forKey: SelectPrinterViewController.KEY_NAME)
            defaults.removeObject(forKey: SelectPrinterViewController.KEY_MAC_ADDRESS)
            defaults.removeObject(forKey: SelectPrinterViewController.KEY_TYPE)
        }
        selectedPrinter = printer
    }
    
    private func resetPrinter() {
        printers.removeAll()
        DispatchQueue.main.async {
            self.searchedPrinters.reloadData();
        }
    }
    
    public static func getSelectedPrinter() -> NPrinter? {
        var result: NPrinter?
        
        let defaults = UserDefaults.standard
        let name = defaults.string(forKey: KEY_NAME) ?? ""
        let macAddress = defaults.string(forKey: KEY_MAC_ADDRESS) ?? ""
        let typeValue = defaults.integer(forKey: KEY_TYPE) 
        
        var type = NPrinterType.none
        switch typeValue {
        case NPrinterType.none.rawValue:
            type = .none
        case NPrinterType.nemonic.rawValue:
            type = .nemonic
        case NPrinterType.nemonicLabel.rawValue:
            type = .nemonicLabel
        case NPrinterType.nemonicMini.rawValue:
            type = .nemonicMini
        case NPrinterType.nemonicMip201.rawValue:
            type = .nemonicMip201
        default:
            type = .none
        }
        
        result = NPrinter(type: type, name: name, macAddress: macAddress)
        
        return result
    }
}

extension SelectPrinterViewController: UITableViewDataSource, UITableViewDelegate {
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return printers.count
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let result: SearchedPrinterViewCell = tableView.dequeueReusableCell(withIdentifier: "SearchedPrinter", for: indexPath) as! SearchedPrinterViewCell
        
        let printer = printers[indexPath.row]
        let name = printer.getName()
        let macAddress = printer.getMacAddress()
        
        result.setInfo(name: name + "(" + macAddress + ")")
        
        return result
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        let printer = printers[indexPath.row]
        updateSelectedPrinter(printer)
    }
}

extension SelectPrinterViewController: NPrinterScanControllerDelegate {
    public func deviceFound(_ printer: NPrinter) {
        addPrinter(printer)
    }
}
